// Copyright 2024 Peter Dimov
// Distributed under the Boost Software License, Version 1.0.
// https://www.boost.org/LICENSE_1_0.txt

#include <boost/uuid/detail/chacha20.hpp>
#include <boost/uuid/detail/endian.hpp>
#include <boost/core/lightweight_test.hpp>
#include <algorithm>
#include <cstdint>

int main()
{
    namespace detail = boost::uuids::detail;

    // https://datatracker.ietf.org/doc/html/draft-strombergson-chacha-test-vectors-01

    {
        // TC1, all zero
        detail::chacha20_12 gen;

        std::uint32_t stream[ 32 ];
        std::generate( stream, stream + 32, gen );

        std::uint8_t expected[ 256 ] =
        {
            0x9b, 0xf4, 0x9a, 0x6a, 0x07, 0x55, 0xf9, 0x53,
            0x81, 0x1f, 0xce, 0x12, 0x5f, 0x26, 0x83, 0xd5,
            0x04, 0x29, 0xc3, 0xbb, 0x49, 0xe0, 0x74, 0x14,
            0x7e, 0x00, 0x89, 0xa5, 0x2e, 0xae, 0x15, 0x5f,
            0x05, 0x64, 0xf8, 0x79, 0xd2, 0x7a, 0xe3, 0xc0,
            0x2c, 0xe8, 0x28, 0x34, 0xac, 0xfa, 0x8c, 0x79,
            0x3a, 0x62, 0x9f, 0x2c, 0xa0, 0xde, 0x69, 0x19,
            0x61, 0x0b, 0xe8, 0x2f, 0x41, 0x13, 0x26, 0xbe,
            0x0b, 0xd5, 0x88, 0x41, 0x20, 0x3e, 0x74, 0xfe,
            0x86, 0xfc, 0x71, 0x33, 0x8c, 0xe0, 0x17, 0x3d,
            0xc6, 0x28, 0xeb, 0xb7, 0x19, 0xbd, 0xcb, 0xcc,
            0x15, 0x15, 0x85, 0x21, 0x4c, 0xc0, 0x89, 0xb4,
            0x42, 0x25, 0x8d, 0xcd, 0xa1, 0x4c, 0xf1, 0x11,
            0xc6, 0x02, 0xb8, 0x97, 0x1b, 0x8c, 0xc8, 0x43,
            0xe9, 0x1e, 0x46, 0xca, 0x90, 0x51, 0x51, 0xc0,
            0x27, 0x44, 0xa6, 0xb0, 0x17, 0xe6, 0x93, 0x16,
        };

        for( int i = 0; i < 32; ++i )
        {
            std::uint32_t exp = detail::load_little_u32( expected + i * 4 );
            BOOST_TEST_EQ( stream[ i ], exp );
        }
    }

    {
        // TC2, key { 0x01 }, nonce zero
        detail::chacha20_12 gen( { 0x01 }, {} );

        std::uint32_t stream[ 32 ];
        std::generate( stream, stream + 32, gen );

        std::uint8_t expected[ 256 ] =
        {
            0x12, 0x05, 0x6e, 0x59, 0x5d, 0x56, 0xb0, 0xf6,
            0xee, 0xf0, 0x90, 0xf0, 0xcd, 0x25, 0xa2, 0x09,
            0x49, 0x24, 0x8c, 0x27, 0x90, 0x52, 0x5d, 0x0f,
            0x93, 0x02, 0x18, 0xff, 0x0b, 0x4d, 0xdd, 0x10,
            0xa6, 0x00, 0x22, 0x39, 0xd9, 0xa4, 0x54, 0xe2,
            0x9e, 0x10, 0x7a, 0x7d, 0x06, 0xfe, 0xfd, 0xfe,
            0xf0, 0x21, 0x0f, 0xeb, 0xa0, 0x44, 0xf9, 0xf2,
            0x9b, 0x17, 0x72, 0xc9, 0x60, 0xdc, 0x29, 0xc0,
            0x0c, 0x73, 0x66, 0xc5, 0xcb, 0xc6, 0x04, 0x24,
            0x0e, 0x66, 0x5e, 0xb0, 0x2a, 0x69, 0x37, 0x2a,
            0x7a, 0xf9, 0x79, 0xb2, 0x6f, 0xbb, 0x78, 0x09,
            0x2a, 0xc7, 0xc4, 0xb8, 0x80, 0x29, 0xa7, 0xc8,
            0x54, 0x51, 0x3b, 0xc2, 0x17, 0xbb, 0xfc, 0x7d,
            0x90, 0x43, 0x2e, 0x30, 0x8e, 0xba, 0x15, 0xaf,
            0xc6, 0x5a, 0xeb, 0x48, 0xef, 0x10, 0x0d, 0x56,
            0x01, 0xe6, 0xaf, 0xba, 0x25, 0x71, 0x17, 0xa9,
        };

        for( int i = 0; i < 32; ++i )
        {
            std::uint32_t exp = detail::load_little_u32( expected + i * 4 );
            BOOST_TEST_EQ( stream[ i ], exp );
        }
    }

    {
        // TC3, key zero, nonce { 0x01 }
        detail::chacha20_12 gen( {}, { 0x01 } );

        std::uint32_t stream[ 32 ];
        std::generate( stream, stream + 32, gen );

        std::uint8_t expected[ 256 ] =
        {
            0x64, 0xb8, 0xbd, 0xf8, 0x7b, 0x82, 0x8c, 0x4b,
            0x6d, 0xba, 0xf7, 0xef, 0x69, 0x8d, 0xe0, 0x3d,
            0xf8, 0xb3, 0x3f, 0x63, 0x57, 0x14, 0x41, 0x8f,
            0x98, 0x36, 0xad, 0xe5, 0x9b, 0xe1, 0x29, 0x69,
            0x46, 0xc9, 0x53, 0xa0, 0xf3, 0x8e, 0xcf, 0xfc,
            0x9e, 0xcb, 0x98, 0xe8, 0x1d, 0x5d, 0x99, 0xa5,
            0xed, 0xfc, 0x8f, 0x9a, 0x0a, 0x45, 0xb9, 0xe4,
            0x1e, 0xf3, 0xb3, 0x1f, 0x02, 0x8f, 0x1d, 0x0f,
            0x55, 0x9d, 0xb4, 0xa7, 0xf2, 0x22, 0xc4, 0x42,
            0xfe, 0x23, 0xb9, 0xa2, 0x59, 0x6a, 0x88, 0x28,
            0x51, 0x22, 0xee, 0x4f, 0x13, 0x63, 0x89, 0x6e,
            0xa7, 0x7c, 0xa1, 0x50, 0x91, 0x2a, 0xc7, 0x23,
            0xbf, 0xf0, 0x4b, 0x02, 0x6a, 0x2f, 0x80, 0x7e,
            0x03, 0xb2, 0x9c, 0x02, 0x07, 0x7d, 0x7b, 0x06,
            0xfc, 0x1a, 0xb9, 0x82, 0x7c, 0x13, 0xc8, 0x01,
            0x3a, 0x6d, 0x83, 0xbd, 0x3b, 0x52, 0xa2, 0x6f,
        };

        for( int i = 0; i < 32; ++i )
        {
            std::uint32_t exp = detail::load_little_u32( expected + i * 4 );
            BOOST_TEST_EQ( stream[ i ], exp );
        }
    }

    return boost::report_errors();
}
