# Boost.Redis

Boost.Redis is a high-level [Redis](https://redis.io/) client library built on top of
[Boost.Asio](https://www.boost.org/doc/libs/latest/doc/html/boost_asio.html)
that implements the Redis protocol
[RESP3](https://github.com/redis/redis-specifications/blob/master/protocol/RESP3.md).

Full documentation is [here](https://www.boost.org/doc/libs/master/libs/redis/index.html).

## Requirements

The requirements for using Boost.Redis are:

* Boost 1.84 or higher. Boost.Redis is included in Boost installations since Boost 1.84.
* C++17 or higher. Supported compilers include gcc 11 and later, clang 11 and later, and Visual Studio 16 (2019) and later.
* Redis 6 or higher (must support RESP3).
* OpenSSL.

The documentation assumes basic-level knowledge about [Redis](https://redis.io/docs/) and [Boost.Asio](https://www.boost.org/doc/libs/latest/doc/html/boost_asio.html).

## Building the library

To use the library it is necessary to include the following:

```cpp
#include <boost/redis/src.hpp>
```

in exactly one source file in your applications. Otherwise, the library is header-only.

Boost.Redis unconditionally requires OpenSSL. Targets using Boost.Redis need to link
to the OpenSSL libraries.

## Tutorial

The code below uses a short-lived connection to
[ping](https://redis.io/commands/ping/) a Redis server:


```cpp
#include <boost/redis/connection.hpp>
#include <boost/asio/co_spawn.hpp>
#include <boost/asio/consign.hpp>
#include <boost/asio/detached.hpp>
#include <iostream>

namespace net = boost::asio;
using boost::redis::request;
using boost::redis::response;
using boost::redis::config;
using boost::redis::connection;

auto co_main(config const& cfg) -> net::awaitable<void>
{
   auto conn = std::make_shared<connection>(co_await net::this_coro::executor);
   conn->async_run(cfg, {}, net::consign(net::detached, conn));

   // A request containing only a ping command.
   request req;
   req.push("PING", "Hello world");

   // Response object.
   response<std::string> resp;

   // Executes the request.
   co_await conn->async_exec(req, resp);
   conn->cancel();

   std::cout << "PING: " << std::get<0>(resp).value() << std::endl;
}
```

The roles played by the `async_run` and `async_exec` functions are:

* `connection::async_exec`: executes the commands contained in the
  request and stores the individual responses in the response object. Can
  be called from multiple places in your code concurrently.
* `connection::async_run`: keeps the connection healthy. It takes care of hostname resolution, session establishment, health-checks, reconnection and coordination of low-level read and write operations. It should be called only once per connection, regardless of the number of requests to execute.

## Server pushes

Redis servers can also send a variety of pushes to the client. Some of
them are:

* [Pubsub messages](https://redis.io/docs/manual/pubsub/).
* [Keyspace notifications](https://redis.io/docs/manual/keyspace-notifications/).
* [Client-side caching](https://redis.io/docs/manual/client-side-caching/).

The connection class supports server pushes by means of the
`connection::async_receive2` function, which can be
called in the same connection that is being used to execute commands.
The coroutine below shows how to use it


```cpp
auto
receiver(std::shared_ptr<connection> conn) -> net::awaitable<void>
{
   request req;
   req.push("SUBSCRIBE", "channel");

   flat_tree resp;
   conn->set_receive_response(resp);

   // Loop while reconnection is enabled
   while (conn->will_reconnect()) {

      // Reconnect to channels.
      co_await conn->async_exec(req);

      // Loop reading Redis pushes.
      for (error_code ec;;) {
         co_await conn->async_receive2(resp, redirect_error(ec));
         if (ec)
            break; // Connection lost, break so we can reconnect to channels.

         // Use the response resp in some way and then clear it.
         ...

         resp.clear();
      }
   }
}
```

## Further reading

Full documentation is [here](https://www.boost.org/doc/libs/master/libs/redis/index.html).
